#include "tDisplay.h"

const uint8_t tDisplay::_digitTable[20] = {
    0xC0, 0xF9, 0xA4, 0xB0, 0x99, 0x92, 0x82, 0xF8, 0x80, 0x90,  // 0-9
    0x40, 0x79, 0x24, 0x30, 0x19, 0x12, 0x02, 0x78, 0x00, 0x10   // 0.-9.
};

tDisplay::tDisplay(uint8_t bit1, uint8_t bit2, uint8_t bit3, uint8_t bit4, 
                   uint8_t stcp, uint8_t shcp, uint8_t data) {
    _bitPins[0] = bit1;
    _bitPins[1] = bit2;
    _bitPins[2] = bit3;
    _bitPins[3] = bit4;
    _stcpPin = stcp;
    _shcpPin = shcp;
    _dataPin = data;
}

void tDisplay::begin() {
    for (int i = 0; i < 4; i++) {
        pinMode(_bitPins[i], OUTPUT);
        digitalWrite(_bitPins[i], LOW);
    }
    pinMode(_stcpPin, OUTPUT);
    pinMode(_shcpPin, OUTPUT);
    pinMode(_dataPin, OUTPUT);
    
    digitalWrite(_stcpPin, LOW);
    digitalWrite(_shcpPin, LOW);
    turnOffAll();
}

void tDisplay::displayNumber(float number) {
    _setDisplayData(number);
}

void tDisplay::displayInt(int number) {
    _setIntDisplayData(number);
}

void tDisplay::turnOffAll() {
    for (int i = 0; i < 4; i++) {
        digitalWrite(_bitPins[i], LOW);
    }
}

void tDisplay::clear() {
    for (int i = 0; i < 4; i++) {
        _currentDigits[i] = 0;
        _decimalPoints[i] = false;
    }
    turnOffAll();
}

void tDisplay::update() {
    if (millis() - _lastUpdateTime > 1) {
        _lastUpdateTime = millis();
        _displayCurrentDigit();
        _currentPosition = (_currentPosition + 1) % 4;
    }
}

// 中断驱动的逐位扫描：不清空所有位，只关闭上一位再点亮当前位，减少黑屏间隙
// 去除中断版扫描，恢复 update 驱动

void tDisplay::_shiftData(uint8_t data) {
    digitalWrite(_stcpPin, LOW);
    shiftOut(_dataPin, _shcpPin, MSBFIRST, data);
    digitalWrite(_stcpPin, HIGH);
}

void tDisplay::_displayCurrentDigit() {
    turnOffAll();
    
    uint8_t digit = _currentDigits[_currentPosition];
    bool showDot = _decimalPoints[_currentPosition];
    
    if (digit > 9) digit = 9;
    
    uint8_t segmentData;
    if (showDot) {
        segmentData = _digitTable[digit + 10];
    } else {
        segmentData = _digitTable[digit];
    }
    
    _shiftData(segmentData);
    digitalWrite(_bitPins[_currentPosition], HIGH);
}

void tDisplay::_setDisplayData(float number) {
    // 重置显示数据
    for (int i = 0; i < 4; i++) {
        _currentDigits[i] = 0;
        _decimalPoints[i] = false;
    }
    
    // 处理特殊情况
    if (number < 0) {
        // 负数处理，这里简单显示为0
        _currentDigits[0] = 0;
        return;
    }
    
    if (number >= 1000) {
        // 1000-9999: 显示为整数，无小数点
        int num = (int)(number + 0.5);
        _currentDigits[0] = num / 1000;
        _currentDigits[1] = (num % 1000) / 100;
        _currentDigits[2] = (num % 100) / 10;
        _currentDigits[3] = num % 10;
    }
    else if (number >= 100) {
        // 100-999.9: 格式为123.4
        int integerPart = (int)number;
        _currentDigits[0] = integerPart / 100;
        _currentDigits[1] = (integerPart % 100) / 10;
        _currentDigits[2] = integerPart % 10;
        
        // 获取小数部分的第一位
        float fractional = number - integerPart;
        _currentDigits[3] = (int)(fractional * 10 + 0.5);
        
        _decimalPoints[2] = true; // 小数点在第三位后面
    }
    else if (number >= 10) {
        // 10-99.99: 格式为12.34
        int integerPart = (int)number;
        _currentDigits[0] = integerPart / 10;
        _currentDigits[1] = integerPart % 10;
        
        // 获取小数部分
        float fractional = number - integerPart;
        _currentDigits[2] = (int)(fractional * 10 + 0.5);
        _currentDigits[3] = (int)(fractional * 100 + 0.5) % 10;
        
        _decimalPoints[1] = true; // 小数点在第二位后面
    }
    else if (number >= 1) {
        // 1-9.999: 格式为1.234
        int integerPart = (int)number;
        _currentDigits[0] = integerPart;
        
        // 获取小数部分
        float fractional = number - integerPart;
        _currentDigits[1] = (int)(fractional * 10 + 0.5);
        _currentDigits[2] = (int)(fractional * 100 + 0.5) % 10;
        _currentDigits[3] = (int)(fractional * 1000 + 0.5) % 10;
        
        _decimalPoints[0] = true; // 小数点在第一位后面
    }
    else if (number >= 0) {
        // 0-0.999: 格式为0.123
        _currentDigits[0] = 0;
        
        float fractional = number;
        _currentDigits[1] = (int)(fractional * 10 + 0.5);
        _currentDigits[2] = (int)(fractional * 100 + 0.5) % 10;
        _currentDigits[3] = (int)(fractional * 1000 + 0.5) % 10;
        
        _decimalPoints[0] = true; // 小数点在第一位后面
    }
    
    // 边界检查，确保数字在0-9范围内
    for (int i = 0; i < 4; i++) {
        if (_currentDigits[i] > 9) _currentDigits[i] = 9;
        if (_currentDigits[i] < 0) _currentDigits[i] = 0;
    }
}

void tDisplay::_setIntDisplayData(int number) {
    number = constrain(number, 0, 9999);
    
    _currentDigits[0] = number / 1000;
    _currentDigits[1] = (number % 1000) / 100;
    _currentDigits[2] = (number % 100) / 10;
    _currentDigits[3] = number % 10;
    
    for (int i = 0; i < 4; i++) {
        _decimalPoints[i] = false;
        if (_currentDigits[i] > 9) _currentDigits[i] = 9;
        if (_currentDigits[i] < 0) _currentDigits[i] = 0;
    }
}