#include <Arduino.h>
#include <Wire.h>
#include <radio.h>
#include <RDA5807M.h>

#include <RDSParser.h>
#include "tDisplay.h"
#include "OneButton.h"
#include <EEPROM.h> // 增加 EEPROM 用于自动保存频率

// #define IR_USE_TIMER1
// #include <IRremote.h>  //添加这个代码就出问题
#include "IRremote.hpp" // include the library
// #include "IRremote.h"

#define IR_RECEIVE_PIN 2

// 红外遥控命令定义
#define IR_CH_D 0x45       // 频道下降
#define IR_CH 0x46         // 存储当前频道
#define IR_CH_U 0x47       // 频道上升
#define IR_PREV 0x44       // 上一个存储频道
#define IR_NEXT 0x40       // 下一个存储频道
#define IR_PLAY_PAUSE 0x43 // 自动搜索频道/停止搜索
#define IR_VOL_D 0x7       // 音量减小
#define IR_VOL_U 0x15      // 音量增加
#define IR_EQ 0x9          // 静音切换

#define IR_0 0x16   // 数字0
#define IR_100 0x19 // 数字100
#define IR_200 0xD  // 数字200
#define IR_1 0xC    // 数字1
#define IR_2 0x18   // 数字2
#define IR_3 0x5E   // 数字3
#define IR_4 0x8    // 数字4
#define IR_5 0x1C   // 数字5
#define IR_6 0x5A   // 数字6
#define IR_7 0x42   // 数字7
#define IR_8 0x52   // 数字8
#define IR_9 0x4A   // 数字9

#define BIT_CHOICE_1 10
#define BIT_CHOICE_2 9
#define BIT_CHOICE_3 8
#define BIT_CHOICE_4 7

#define STCP_PIN 11
#define SHCP_PIN 13
#define DATA_PIN 12

RDA5807M radio;
RDSParser rds;

enum RADIO_STATE
{
  STATE_PARSECOMMAND, ///< waiting for a new command character.
  STATE_PARSEINT,     ///< waiting for digits for the parameter.
  STATE_EXEC          ///< executing the command.
};

RADIO_STATE kbState;
char kbCommand;
int16_t kbValue;

uint16_t g_block1;
bool radioDebug = false;
bool lowLevelDebug = false;
// Avoid dynamic String to save SRAM and fragmentation on AVR
char lastServiceName[9] = ""; // 8 chars + null terminator for RDS PS name

RADIO_FREQ _fSave, _fLast = 0;

tDisplay display(BIT_CHOICE_1, BIT_CHOICE_2, BIT_CHOICE_3, BIT_CHOICE_4, STCP_PIN, SHCP_PIN, DATA_PIN);

// 四个前面板按钮：音量+/音量-/频率+/频率-
OneButton buttonVolumeUp;      // 音量+
OneButton buttonVolumeDown;    // 音量-
OneButton buttonFrequencyUp;   // 频率+
OneButton buttonFrequencyDown; // 频率-

// 频率长按快速调节状态
bool freqUpHoldActive = false;
bool freqDownHoldActive = false;
unsigned long lastFreqHoldMillis = 0;
const uint16_t FAST_FREQ_INTERVAL_MS = 120; // 长按时步进间隔（毫秒）

// 显示模式：显示频率或显示音量（音量显示2秒后恢复频率）
enum DisplayMode
{
  DISPLAY_FREQ,
  DISPLAY_VOLUME
};
DisplayMode gDisplayMode = DISPLAY_FREQ;
unsigned long displayHoldUntil = 0;

// 自动频率保存（稳定5秒）
unsigned long lastFreqChangeMillis = 0;
RADIO_FREQ lastObservedFreq = 0;
bool lastFreqSaved = true;               // 初始认为已保存
const unsigned long AUTO_SAVE_MS = 5000; // 5秒后保存
const uint8_t EE_FREQ_ADDR_LO = 0;       // EEPROM 地址低字节
const uint8_t EE_FREQ_ADDR_HI = 1;       // EEPROM 地址高字节

// 预置电台列表（长度10）EEPROM 布局：
// [0..1] 最近频率（已使用）
// [2]    预置数量 presetCount
// [3]    当前索引 presetIndex
// [4..23] 10 * 2 bytes 频率值（低字节在前）
static const uint8_t EE_PRESET_COUNT_ADDR = 2;
static const uint8_t EE_PRESET_INDEX_ADDR = 3;
static const uint8_t EE_PRESET_DATA_ADDR = 4;
static const uint8_t MAX_PRESETS = 10;
uint8_t presetCount = 0;
uint8_t presetIndex = 0;
RADIO_FREQ presets[MAX_PRESETS];

inline bool validFreq(RADIO_FREQ f)
{
  return f >= radio.getMinFrequency() && f <= radio.getMaxFrequency();
}

void savePresets()
{
  // 保存当前预置列表到 EEPROM。
  // 数据布局：计数、当前索引、10*频率值（小端）。
  EEPROM.update(EE_PRESET_COUNT_ADDR, presetCount);
  EEPROM.update(EE_PRESET_INDEX_ADDR, presetIndex);
  uint16_t addr = EE_PRESET_DATA_ADDR;
  for (uint8_t i = 0; i < MAX_PRESETS; i++)
  {
    RADIO_FREQ fr = (i < presetCount) ? presets[i] : 0;
    EEPROM.update(addr++, (uint8_t)(fr & 0xFF));
    EEPROM.update(addr++, (uint8_t)(fr >> 8));
  }
}

void loadPresets()
{
  // 从 EEPROM 读取预置列表到内存，并做基本合法性校验。
  presetCount = EEPROM.read(EE_PRESET_COUNT_ADDR);
  presetIndex = EEPROM.read(EE_PRESET_INDEX_ADDR);
  if (presetCount > MAX_PRESETS)
    presetCount = 0;
  uint16_t addr = EE_PRESET_DATA_ADDR;
  for (uint8_t i = 0; i < MAX_PRESETS; i++)
  {
    uint16_t lo = EEPROM.read(addr++);
    uint16_t hi = EEPROM.read(addr++);
    presets[i] = (hi << 8) | lo;
  }
  if (presetCount > 0)
  {
    if (presetIndex >= presetCount || !validFreq(presets[presetIndex]))
      presetIndex = 0;
    for (uint8_t i = 0; i < presetCount; i++)
    {
      if (!validFreq(presets[i]))
        presets[i] = radio.getMinFrequency();
    }
  }
}

void storeCurrentPreset()
{
  // 将当前频率存入预置列表：若已存在则更新索引；未存在且未满则追加；已满覆盖当前索引。
  RADIO_FREQ f = radio.getFrequency();
  // 已存在则只更新索引
  for (uint8_t i = 0; i < presetCount; i++)
  {
    Serial.println(presets[i]);
    if (presets[i] == f)
    {
      presetIndex = i;
      savePresets();
      return;
    }
  }
  if (presetCount < MAX_PRESETS)
  {
    presets[presetCount] = f;
    presetIndex = presetCount;
    presetCount++;
  }
  else
  {
    // 列表已满则覆盖当前索引
    presets[presetIndex] = f;
  }
  savePresets();
}

void gotoPreset(uint8_t idx)
{
  // 跳转到指定预置索引（循环取模），并切换显示为频率。
  if (presetCount == 0)
    return;
  idx %= presetCount;
  presetIndex = idx;
  radio.setFrequency(presets[presetIndex]);
  gDisplayMode = DISPLAY_FREQ;
}

void gotoNextPreset(int dir)
{
  // 根据方向 dir=+1/-1 切换到下一个或上一个预置（循环）。
  if (presetCount == 0)
    return;
  int n = (int)presetIndex + dir;
  if (n < 0)
    n = presetCount - 1;
  if (n >= presetCount)
    n = 0;
  gotoPreset((uint8_t)n);
}

inline void saveCurrentFrequency()
{
  // 自动保存当前频率到 EEPROM（用于上次开机恢复），防止频繁写入由逻辑控制调用频率。
  RADIO_FREQ f = radio.getFrequency();
  EEPROM.update(EE_FREQ_ADDR_LO, (uint8_t)(f & 0xFF));
  EEPROM.update(EE_FREQ_ADDR_HI, (uint8_t)(f >> 8));
  lastFreqSaved = true;
}

void RDS_process(uint16_t block1, uint16_t block2, uint16_t block3, uint16_t block4)
{
  // RDS 数据接收回调：将四个区块传给解析器。
  g_block1 = block1;
  rds.processData(block1, block2, block3, block4);
}

void DisplayServiceName(const char *name)
{
  // 显示并缓存电台服务名称（去除全空格的无效名称）。
  bool found = false;

  for (uint8_t n = 0; n < 8; n++)
    if (name[n] != ' ')
      found = true;

  if (found)
  {
    // Copy up to 8 chars of service name into fixed buffer
    strncpy(lastServiceName, name, 8);
    lastServiceName[8] = '\0';
    Serial.print(F("Sender:<"));
    Serial.print(name);
    Serial.println('>');
  }
}

void PrintScanInfo(RADIO_INFO *ri)
{
  // 打印当前频率的接收状态：频率、信号强度、调谐、立体声、RDS 标志。
  char sFreq[12];
  radio.formatFrequency(sFreq, sizeof(sFreq));
  Serial.print(sFreq);
  Serial.print(' ');

  Serial.print(ri->rssi);
  Serial.print(' ');
  Serial.print(ri->tuned ? 'T' : '-');
  Serial.print(ri->stereo ? 'S' : '-');
  Serial.print(ri->rds ? 'R' : '-');
  Serial.println();
}

void runSerialCommand(char cmd, int16_t value)
{
  // 串口命令解释器：解析用户输入的命令字符并执行对应的收音机操作。
  // 支持调音量/模式切换/扫描/RDS 查询/调谐等功能。
  unsigned long startSeek; // after 300 msec must be tuned. after 500 msec must have RDS.
  RADIO_FREQ fSave, fLast = 0;
  RADIO_FREQ f = radio.getMinFrequency();
  RADIO_FREQ fMax = radio.getMaxFrequency();
  RADIO_INFO ri;

  if ((cmd == '\n') || (cmd == '\r'))
  {
    return;
  }

  Serial.print(F("do:"));
  Serial.println(cmd);

  if (cmd == '?')
  {
    Serial.println();
    Serial.println(F("? Help"));
    Serial.println(F("+ increase volume"));
    Serial.println(F("- decrease volume"));
    Serial.println(F("1 scan Frequency + Data"));
    Serial.println(F("2 scan version 2"));
    Serial.println(F("3 scan RDS stations"));
    Serial.println(F(". scan up   : scan up to next sender"));
    Serial.println(F(", scan down ; scan down to next sender"));
    Serial.println(F("i station status"));
    Serial.println(F("s mono/stereo mode"));
    Serial.println(F("b bass boost"));
    Serial.println(F("m mute/unmute"));
    Serial.println(F("u soft mute/unmute"));
    Serial.println(F("x debug..."));
    Serial.println(F("y toggle Debug Messages..."));
    Serial.println(F("* toggle i2c debug output"));
  }
  else if (cmd == '+')
  {
    radio.setMute(false);
    int v = radio.getVolume();
    if (v < 15)
      radio.setVolume(++v);
    // 显示音量并在2秒后恢复频率
    gDisplayMode = DISPLAY_VOLUME;
    displayHoldUntil = millis() + 2000;
  }
  else if (cmd == '-')
  {
    radio.setMute(false);
    int v = radio.getVolume();
    if (v > 0)
      radio.setVolume(--v);
    gDisplayMode = DISPLAY_VOLUME;
    displayHoldUntil = millis() + 2000;
  }
  else if (cmd == 'm')
  {
    radio.setMute(!radio.getMute());
  }
  else if (cmd == 'u')
  {
    radio.setSoftMute(!radio.getSoftMute());
  }
  else if (cmd == 's')
  {
    radio.setMono(!radio.getMono());
  }
  else if (cmd == 'b')
  {
    radio.setBassBoost(!radio.getBassBoost());
  }
  else if (cmd == '1')
  {
    Serial.println(F("Scanning all available frequencies... (1)"));
    fSave = radio.getFrequency();

    while (f <= fMax)
    {
      radio.setFrequency(f);
      delay(100);
      radio.getRadioInfo(&ri);

      if (true)
      {
        RADIO_FREQ f = radio.getFrequency();
        Serial.print(f);
        Serial.print(',');
        Serial.print(ri.stereo);
        Serial.print(',');
        Serial.print(ri.rds);
        Serial.print(',');
        Serial.println(ri.rssi);
      }

      // tune up by 1 step
      f += radio.getFrequencyStep();
    } // while
    Serial.println(F("done."));
    radio.setFrequency(fSave);
  }
  else if (cmd == '2')
  {
    Serial.println(F("Seeking all frequencies... (2)"));
    fSave = radio.getFrequency();

    // start Scan
    radio.setFrequency(f);

    while (f <= fMax)
    {
      radio.seekUp(true);
      delay(100); //
      startSeek = millis();

      // wait for seek complete
      do
      {
        radio.getRadioInfo(&ri);
      } while ((!ri.tuned) && (startSeek + 600 > millis()));

      // check frequency
      f = radio.getFrequency();
      if (f < fLast)
      {
        break;
      }
      fLast = f;

      if (true)
      { // every frequency detected by builtin scan

        radio.getRadioInfo(&ri);
        PrintScanInfo(&ri);
      }
    }
    Serial.println(F("done."));
    radio.setFrequency(fSave);
  }
  else if (cmd == '3')
  {
    Serial.println(F("Seeking all RDS senders..."));
    fSave = radio.getFrequency();

    // start Scan
    radio.setFrequency(f);

    // start Simple Scan: all channels
    while (f <= fMax)
    {
      radio.setFrequency(f);
      lastServiceName[0] = '\0';
      delay(100);
      radio.getRadioInfo(&ri);
      if (ri.rssi >= 32)
      { // si4703 usable threshold value
        if (ri.rds)
        {
          radio.checkRDS();
          PrintScanInfo(&ri);
          startSeek = millis();

          // wait 3 secs for sender name
          do
          {
            radio.checkRDS();
            delay(30);
          } while (lastServiceName[0] == '\0' && (startSeek + 6000 > millis()));
        } // if
      }
      // tune up by 1 step
      f += radio.getFrequencyStep();
    }
    Serial.println(F("done."));
    radio.setFrequency(fSave);
  }
  else if (cmd == 'f')
  {
    radio.setFrequency(value);
    gDisplayMode = DISPLAY_FREQ;
  }

  else if (cmd == '.')
  {
    radio.seekUp(false);
    gDisplayMode = DISPLAY_FREQ;
  }
  else if (cmd == ':')
  {
    radio.seekUp(true);
    gDisplayMode = DISPLAY_FREQ;
  }
  else if (cmd == ',')
  {
    radio.seekDown(false);
    gDisplayMode = DISPLAY_FREQ;
  }
  else if (cmd == ';')
  {
    radio.seekDown(true);
    gDisplayMode = DISPLAY_FREQ;
  }
  else if (cmd == '!')
  {
    // not in help
    RADIO_FREQ f = radio.getFrequency();
    if (value == 0)
    {
      radio.term();
    }
    else if (value == 1)
    {
      radio.init();
      radio.setBandFrequency(RADIO_BAND_FM, f);
    }
  }
  else if (cmd == 'i')
  {
    // info
    char s[12];
    radio.formatFrequency(s, sizeof(s));
    Serial.print(F("Station:"));
    Serial.println(s);
    Serial.print(F("Radio:"));
    radio.debugRadioInfo();
    Serial.print(F("Audio:"));
    radio.debugAudioInfo();
  }
  else if (cmd == 'x')
  {
    radio.debugStatus(); // print chip specific data.
  }
  else if (cmd == 'y')
  {
    radioDebug = !radioDebug;
    radio.debugEnable(radioDebug);
  }
  else if (cmd == '*')
  {
    lowLevelDebug = !lowLevelDebug;
    radio._wireDebug(lowLevelDebug);
  }
}

// 按钮长按处理
void onLongPressStart(void *btn)
{
  // 按钮长按开始的示例回调（目前仅打印提示）。
  Serial.println("长按开始 - 解除设备绑定");
}

void setup()
{
  // 初始化硬件与软件状态：串口、按钮、收音机芯片、预置与频率恢复、RDS、显示、红外接收。
  Serial.begin(115200);
  Serial.println(F("ScanRadio..."));
  delay(200);
  pinMode(3, INPUT_PULLUP);
  pinMode(4, INPUT_PULLUP);
  pinMode(5, INPUT_PULLUP);
  pinMode(6, INPUT_PULLUP);

  // 初始化按钮（音量+/音量-/频率+/频率-）
  buttonVolumeUp.setup(5, INPUT_PULLUP, true);
  buttonVolumeUp.attachLongPressStart(onLongPressStart, &buttonVolumeUp);
  buttonVolumeUp.setLongPressIntervalMs(2000);
  buttonVolumeUp.attachClick([]()
                             { runSerialCommand('+', 0); });

  buttonVolumeDown.setup(6, INPUT_PULLUP, true);
  buttonVolumeDown.attachLongPressStart(onLongPressStart, &buttonVolumeDown);
  buttonVolumeDown.setLongPressIntervalMs(2000);
  buttonVolumeDown.attachClick([]()
                               { runSerialCommand('-', 0); });

  buttonFrequencyUp.setup(3, INPUT_PULLUP, true);
  buttonFrequencyUp.attachLongPressStart([](void *)
                                         { freqUpHoldActive = true; lastFreqHoldMillis = 0; }, nullptr);
  buttonFrequencyUp.attachLongPressStop([](void *)
                                        { freqUpHoldActive = false; }, nullptr);
  buttonFrequencyUp.setLongPressIntervalMs(2000);
  buttonFrequencyUp.attachClick([]()
                                { runSerialCommand('.', 0); });

  buttonFrequencyDown.setup(4, INPUT_PULLUP, true);
  buttonFrequencyDown.attachLongPressStart([](void *)
                                           { freqDownHoldActive = true; lastFreqHoldMillis = 0; }, nullptr);
  buttonFrequencyDown.attachLongPressStop([](void *)
                                          { freqDownHoldActive = false; }, nullptr);
  buttonFrequencyDown.setLongPressIntervalMs(2000);
  buttonFrequencyDown.attachClick([]()
                                  { runSerialCommand(',', 0); });

  delay(200);
  radio.debugEnable(false);
  radio._wireDebug(lowLevelDebug);
  // Set FM Options for Europe
  radio.setup(RADIO_FMSPACING, RADIO_FMSPACING_100);
  radio.setup(RADIO_DEEMPHASIS, RADIO_DEEMPHASIS_50);

  if (!radio.initWire(Wire))
  {
    Serial.println(F("no radio chip found."));
    delay(4000);
    while (1)
    {
    };
  };

  // 读取上次保存频率（若在允许范围内）否则使用默认 94.80MHz
  loadPresets();
  uint16_t lo = EEPROM.read(EE_FREQ_ADDR_LO);
  uint16_t hi = EEPROM.read(EE_FREQ_ADDR_HI);
  RADIO_FREQ stored = (hi << 8) | lo;
  RADIO_FREQ minF = radio.getMinFrequency();
  RADIO_FREQ maxF = radio.getMaxFrequency();
  if (stored >= minF && stored <= maxF)
  {
    radio.setBandFrequency(RADIO_BAND_FM, stored);
  }
  else if (presetCount > 0 && validFreq(presets[presetIndex]))
  {
    radio.setBandFrequency(RADIO_BAND_FM, presets[presetIndex]);
  }
  else
  {
    radio.setBandFrequency(RADIO_BAND_FM, 9480);
  }
  lastObservedFreq = radio.getFrequency();
  lastFreqChangeMillis = millis();

  radio.setMono(false);
  radio.setMute(false);
  radio.setVolume(2);

  radio.attachReceiveRDS(RDS_process);
  rds.attachServiceNameCallback(DisplayServiceName);

  runSerialCommand('?', 0);
  kbState = STATE_PARSECOMMAND;
  display.begin();

  IrReceiver.begin(IR_RECEIVE_PIN);
}

void loop()
{
  // 主循环：处理串口命令、红外输入、按钮事件、长按快速调频、自动保存频率、以及显示刷新。
  if (Serial.available() > 0)
  {
    char c = Serial.peek();
    if ((kbState == STATE_PARSECOMMAND) && (c < 0x20))
    {
      // ignore unprintable chars
      Serial.read();
    }
    else if (kbState == STATE_PARSECOMMAND)
    {
      // read a command.
      kbCommand = Serial.read();
      kbState = STATE_PARSEINT;
    }
    else if (kbState == STATE_PARSEINT)
    {
      if ((c >= '0') && (c <= '9'))
      {
        // build up the value.
        c = Serial.read();
        kbValue = (kbValue * 10) + (c - '0');
      }
      else
      {
        runSerialCommand(kbCommand, kbValue);
        kbCommand = ' ';
        kbState = STATE_PARSECOMMAND;
        kbValue = 0;
      }
    }
  }

  // check for RDS data
  radio.checkRDS();
  // 更新按钮状态
  buttonFrequencyDown.tick();
  buttonFrequencyUp.tick();
  buttonVolumeDown.tick();
  buttonVolumeUp.tick();

  // 长按快速调频：按住上/下，每隔 FAST_FREQ_INTERVAL_MS 调整一次
  if (freqUpHoldActive && (millis() - lastFreqHoldMillis >= FAST_FREQ_INTERVAL_MS))
  {
    radio.seekUp(false); // 单步向上
    lastFreqHoldMillis = millis();
    // 标记频率变化
    lastObservedFreq = radio.getFrequency();
    lastFreqChangeMillis = millis();
    lastFreqSaved = false;
    gDisplayMode = DISPLAY_FREQ; // 长按调频时立即切到频率显示
  }
  if (freqDownHoldActive && (millis() - lastFreqHoldMillis >= FAST_FREQ_INTERVAL_MS))
  {
    radio.seekDown(false); // 单步向下
    lastFreqHoldMillis = millis();
    lastObservedFreq = radio.getFrequency();
    lastFreqChangeMillis = millis();
    lastFreqSaved = false;
    gDisplayMode = DISPLAY_FREQ;
  }

  if (IrReceiver.decode())
  {

    /*
     * Print a summary of received data
     */
    if (IrReceiver.decodedIRData.protocol == UNKNOWN)
    {
      // Serial.println(F("Received noise or an unknown (or not yet enabled) protocol"));
      // We have an unknown protocol here, print extended info
      // IrReceiver.printIRResultRawFormatted(&Serial, true);

      IrReceiver.resume(); // Do it here, to preserve raw data for printing with printIRResultRawFormatted()
    }
    else
    {
      IrReceiver.resume(); // Early enable receiving of the next IR frame

      // IrReceiver.printIRResultShort(&Serial);
      // IrReceiver.printIRSendUsage(&Serial);
    }
    // Serial.println();

    /*
     * Finally, check the received data and perform actions according to the received command
     */
    if (IrReceiver.decodedIRData.flags & IRDATA_FLAGS_IS_REPEAT)
    {
      Serial.println(F("Repeat received. Here you can repeat the same action as before."));
    }
    else
    {
      uint8_t cmd = IrReceiver.decodedIRData.command;
      if (cmd == IR_CH_D)
      {
        // 单步向下调频
        runSerialCommand(',', 0);
      }
      else if (cmd == IR_CH_U)
      {
        // 单步向上调频
        runSerialCommand('.', 0);
      }
      else if (cmd == IR_VOL_D)
      {
        runSerialCommand('-', 0);
      }
      else if (cmd == IR_VOL_U)
      {
        runSerialCommand('+', 0);
      }
      else if (cmd == IR_CH)
      {
        // 存储当前电台到预置列表（长度10）
        storeCurrentPreset();
      }
      else if (cmd == IR_PREV)
      {
        // 切换到上一个预置
        // gotoNextPreset(-1);
        runSerialCommand(';', 0);
      }
      else if (cmd == IR_NEXT)
      {
        // 切换到下一个预置
        // gotoNextPreset(+1);
        runSerialCommand(':', 0);
      }
      else if (cmd == IR_EQ || cmd == IR_PLAY_PAUSE)
      {
        // 静音
        runSerialCommand('m', 0);
      }
    }
  }
  // 频率变化检测 & 自动保存
  RADIO_FREQ currentFreq = radio.getFrequency();
  if (currentFreq != lastObservedFreq)
  {
    lastObservedFreq = currentFreq;
    lastFreqChangeMillis = millis();
    lastFreqSaved = false;
  }
  else if (!lastFreqSaved && (millis() - lastFreqChangeMillis >= AUTO_SAVE_MS))
  {
    saveCurrentFrequency();
  }

  // 显示逻辑：音量优先显示2秒，其余时间显示频率
  if (gDisplayMode == DISPLAY_VOLUME)
  {
    if (millis() < displayHoldUntil)
    {
      display.displayInt(radio.getVolume());
    }
    else
    {
      gDisplayMode = DISPLAY_FREQ;
    }
  }
  if (gDisplayMode == DISPLAY_FREQ)
  {
    display.displayNumber((int)currentFreq * 0.01);
  }
  display.update();
}
